;Output driver for monochrome 144-pixel (BlinkenLights) projects with 1/18-Mux
;Supported projects:
;  * BlinkstroemAdvanced (by Kai Gossner, 2004)
;  * BlinkstroemAdvanced-Stream (by Kai Gossner & Arne Rossius, 2008)
;Supported controllers: Mega16, probably Mega32
;
;PORTS (ATmega16, BlinkstroemAdvanced[-Stream] only):
;  PA0-7 = LED Col 3..10 (MSB left)
;  PB0-1 = LED Col 2..1 (MSB left)
;  PB2   = SD-Card Detect Switch (low = card is present)
;  PB3   = unused Push-button input (original) / LED Row 7 (BSA-Stream)
;  PB4   = SD-Card D_out (MISO) (resistor 1k)
;  PB5   = SD-Card Clock (voltage divider 1k/2.2k)
;  PB6   = SD-Card D_in (MOSI) (voltage divider 1k/2.2k)
;  PB7   = SD-Card #CS (voltage divider 1k/2.2k)
;  PC0-7 = LED Col 18..11 (MSB left)
;  PD0   = LED Row 1 (original) / RS232 Rx (BSA-Stream)
;  PD1-7 = LED Row 2..8 (MSB bottom)
;
;Frame Timing:
;  Controller frequency   : 16 MHz
;  Timer prescaler        : 256
;  PWM "steps" per column : 35
;  Multiplexing           : 18 columns
;    => 16 MHz / 256 / 35 / 18 = 99.206 Hz (error: -0.794 %)
;
;Example PWM durations (for 144-pixel BlinkstroemAdvanced):
;0, 1, 2, 3, 6, 10, 19, 35  - exponential, base 1.8
;0, 1, 2, 4, 6, 10, 19, 35  - equivalent to old optimized exponential
;0, 1, 3, 6, 11, 18, 26, 35 - Gamma 2.0
;0, 1, 2, 4, 9, 15, 24, 35  - Gamma 2.5
;absolute minimum timestep between two values: 1
;TODO: convert to max. value = 140 (with Clk/64 and min. timestep = 4)

.equ	PWMVAL_0 = 0 ;don't change this value
.equ	PWMVAL_1 = 1
.equ	PWMVAL_2 = 2
.equ	PWMVAL_3 = 4
.equ	PWMVAL_4 = 6
.equ	PWMVAL_5 = 10
.equ	PWMVAL_6 = 19
.equ	PWMVAL_7 = 35 ;don't change this value (timing depends on it)

;===============================================================================

;SD card pins
.equ	USE_HARDWARE_SPI = 0
.equ	SD_PORT = PORTB
.equ	SD_PIN = PINB
.equ	SD_CS = 7
.equ	SD_MOSI = 6
.equ	SD_MISO = 4
.equ	SD_CK = 5
.equ	SDCARD_HAS_DETECT_PIN = 1
.equ	SD_DETECT_PIN = PINB
.equ	SD_DETECT = 2 ;Card Detect switch pin on PORTB

;other settings
.equ	OUT_TIMING_DIV = 256
.equ	DISALLOW_UART_TX = 1
.if (OUT_MOVE_ROW1)
  .equ	DISALLOW_UART = 0
.else
  .equ	DISALLOW_UART = 1
.endif

;===============================================================================

.macro set_pixel
	ld	temp, Z
	sbrs	mux, 0
	swap	temp
	cp	temp, pwm ;carry set if pwm > temp
	ror	R0
	adiw	ZH:ZL, 18/2
.endmacro

;===============================================================================
	
oc0:	;Timer 0 output compare interrupt (PWM steps)
	push	temp
	push	temp2
	_push_w	Z
	push	R0
	push	R1
	in	sreg_backup, SREG
	
	;increment column
	inc	mux
	cpi	mux, 24
	brlo	oc0_colinc_end
	ldi	mux, 6

	;increment PWM value ('pwm' register stores pwm value in high nibble!)
	subi	pwm, -0x30 ;PWM value sequence: 3, 6, 1, 4, 7, 2, 5, (0)
	andi	pwm, 0x70
	brne	oc0_pwminc_end ;zero: end of PWM sequence
	ldi	pwm, 0x30 ;start with first value of PWM sequence
	tick_100hz
oc0_pwminc_end:
	;load start address
	activeframe Z ;max. 6 cycles
	_sts_w	RAM_MuxAddress, Z
oc0_colinc_end:
	
	;determine next interval
	mov	temp, pwm
	swap	temp
	_ldi_w	Z, (FLASH_OFFSET + pwm_duration) * 2
	add	ZL, temp
	adc	ZH, zero
	lpm	temp, Z
	out	OCR0, temp
	
	;load framebuffer address for current column
	_lds_w	Z, RAM_MuxAddress
	
	;set LEDs
	set_pixel ;8 cycles each
	set_pixel
	set_pixel
	set_pixel
	set_pixel
	set_pixel
	set_pixel
	set_pixel
.if (OUT_INVERT_ROWS == 0)
	com	R0
.endif
	subi	ZL, LOW(8 * 18/2)
	sbci	ZH, HIGH(8 * 18/2)
	sbrc	mux, 0
	adiw	ZH:ZL, 1

	;old column off
	in	temp2, PORTB
.if (OUT_INVERT_COLUMNS)
	ldi	temp, 0xFF
	ori	temp2, 0x03
.else
	ldi	temp, 0x00
	andi	temp2, ~0x03
.endif
	out	PORTA, temp
	out	PORTB, temp2
	out	PORTC, temp
	
	;delay until transistor is off
	_sts_w	RAM_MuxAddress, Z
	
	;output new LED data
.if (OUT_MOVE_ROW1)
	andi	temp2, ~0x08
	sbrc	R0, 0
	ori	temp2, 0x08
	ldi	temp, 0x01
	or	R0, temp
.endif
	out	PORTD, R0
	out	PORTB, temp2
	
	;delay until transistors have switched
	mov	R0, mux
	sbrc	mux, 3 ;bit 3 set from 8 (col 3) to 15 (col 10) = cols on PORTA
	com	R0 ;invert (resulting bitmask: 0 -> 0x01 ... 7 -> 0x80)
	;calculate bitmask from lower 3 bits of col (7 -> 0x01 ... 0 -> 0x80)
	ldi	temp, 0x80
	sbrc	R0, 1
	ldi	temp, 0x20
	sbrc	R0, 0
	lsr	temp
	sbrc	R0, 2
	swap	temp
.if (OUT_INVERT_COLUMNS)
	com	temp
.endif
	
	;new column on
	cpi	mux, 8
	brlo	oc0_col_b ;mux == [6, 7]  ->  PORTB
	cpi	mux, 16
	brsh	oc0_col_c ;mux == [16, ..., 23]  ->  PORTC
	out	PORTA, temp ;mux == [8, ..., 15]  ->  PORTA
	rjmp	oc0_end
	
oc0_col_b:
.if (OUT_INVERT_COLUMNS)
	and	temp2, temp
.else
	or	temp2, temp
.endif
	out	PORTB, temp2
	rjmp	oc0_end
	
oc0_col_c:
	out	PORTC, temp

oc0_end:
	out	SREG, sreg_backup
	pop	R1
	pop	R0
	_pop_w	Z
	pop	temp2
	pop	temp
	;return
	reti

;===============================================================================

.macro init_output
	
	;initialize registers
	ldi	mux, 23 ;last column in sequence
	ldi	pwm, 0x50 ;last PWM value in sequence

	;initialize ports
	ldi	temp, 0xFF
	out	DDRA, temp
	out	DDRC, temp
  .if (OUT_INVERT_COLUMNS)
	ldi	temp, 0xFF
  .else
	ldi	temp, 0x00
  .endif
	out	PORTA, temp
	out	PORTC, temp
	
  .if (OUT_MOVE_ROW1)
	ldi	temp, 0xFE ;PD0 = UART Rx input
	ldi	temp2, 1<<SD_CS | 1<<SD_MOSI | 1<<SD_CK | 0x0B ;PB3 = Row 1 out
  .else
	ldi	temp, 0xFF ;PD0 = Row 1 output
	ldi	temp2, 1<<SD_CS | 1<<SD_MOSI | 1<<SD_CK | 0x03 ;PB3 = button in
  .endif
	out	DDRD, temp
	out	DDRB, temp2

  .if (OUT_INVERT_ROWS)
	ldi	temp, 0xFF ;all rows off (high) [and enable pull-up for UART Rx]
  .else
    .if (OUT_MOVE_ROW1)
	ldi	temp, 0x01 ;enable pull-up for UART Rx
    .else
    	ldi	temp, 0x00 ;all rows off (low)
    .endif
  .endif
	out	PORTD, temp

  .if (OUT_INVERT_COLUMNS)
    .if (OUT_MOVE_ROW1 && (OUT_INVERT_ROWS == 0))
	ldi	temp2, 1<<SD_CS | 1<<SD_MOSI | 1<<SD_MISO | 0x07
    .else
	ldi	temp2, 1<<SD_CS | 1<<SD_MOSI | 1<<SD_MISO | 0x0F
    .endif
  .else
    .if (OUT_MOVE_ROW1 && (OUT_INVERT_ROWS == 0))
	ldi	temp2, 1<<SD_CS | 1<<SD_MOSI | 1<<SD_MISO | 0x04
    .else
	ldi	temp2, 1<<SD_CS | 1<<SD_MOSI | 1<<SD_MISO | 0x0C
    .endif
  .endif
	out	PORTB, temp2

.endmacro
