;Output driver for monochrome 520-pixel (ARCADE) projects with 1/20-Mux
;Supported projects:
;  * ArcadeMicro
;Supported controllers: Mega162
;
;PORTS (ATmega162, 520 pixels, SD card using hardware SPI):
;  PA0-7 = LED Col 5..12
;  PB0-3 = LED Col 4..1
;  PB4   = SD-Card #CS (voltage divider 1k/2.2k)
;  PB5   = SD-Card D_in (MOSI) (voltage divider 1k/2.2k)
;  PB6   = SD-Card D_out (MISO) (resistor 1k)
;  PB7   = SD-Card Clock (voltage divider 1k/2.2k)
;  PC0-7 = LED Col 20..13
;  PD0/1 = RS232 Rx/Tx
;  PD2-7 = LED Col 26..21
;  PE0   = SD-Card Detect Switch (low = card is present)
;  PE1   = LED Row shift register data (Q1 = row 20 ... Q20 = row 1)
;  PE2   = LED Row shift register clock
;
;Frame Timing:
;  Controller frequency : 14.7456 MHz
;  Timer prescaler      : 64
;  PWM "steps" per row  : 115
;  Multiplexing         : 20 rows
;    => 14.7456 MHz / 64 / 115 / 20 = ~100.174 Hz (error: ~0.174 %)
;
;Example PWM durations (for 520-pixel projects):
;0, 4, 8, 13, 20, 34, 63, 115 - equivalent to old optimized exponential
;absolute minimum step between two values: 4

.equ	PWMVAL_0 = 0 ;don't change this value
.equ	PWMVAL_1 = 4
.equ	PWMVAL_2 = 8
.equ	PWMVAL_3 = 13
.equ	PWMVAL_4 = 20
.equ	PWMVAL_5 = 34
.equ	PWMVAL_6 = 63
.equ	PWMVAL_7 = 115 ;don't change this value (timing depends on it)

;===============================================================================

;SD card pins
.equ	USE_HARDWARE_SPI = 1
.equ	SD_PORT = PORTB
.equ	SD_CS = 4
.equ	SDCARD_HAS_DETECT_PIN = 1
.equ	SD_DETECT_PIN = PINE
.equ	SD_DETECT = 0 ;Card Detect switch pin on PORTE

;shift register pins (PORTD)
.equ	ROW_DATA = 1
.equ	ROW_CLOCK = 2

;other settings
.equ	OUT_TIMING_DIV = 64
.equ	DISALLOW_UART = 0
.equ	DISALLOW_UART_TX = 0

;===============================================================================

.if (MCU != MCU_MEGA162)
	.error "Output module for ARCADE only works with ATmega162!"
.endif

.macro _set_led_A ;LEDs 26 to 21
	ld	temp, -Z
	cp	temp, pwm
	ror	R0
	swap	temp
	cp	temp, pwm
	ror	R0
.endmacro

.macro _set_led_B ;LEDs 20 to 13
	ld	temp, -Z
	cp	temp, pwm
	ror	R1
	swap	temp
	cp	temp, pwm
	ror	R1
.endmacro

.macro _set_led_C ;LEDs 12 to 5
	ld	temp, -Z
	cp	temp, pwm
	rol	R2
	swap	temp
	cp	temp, pwm
	rol	R2
.endmacro

.macro _set_led_D ;LEDs 4 to 1
	ld	temp2, -Z
	cp	temp2, pwm
	brlo	PC+2
  .if (OUT_INVERT_COLUMNS)
	andi	temp, ~(@0)
  .else
	ori	temp, @0
  .endif
	swap	temp2
	cp	temp2, pwm
	brlo	PC+2
  .if (OUT_INVERT_COLUMNS)
	andi	temp, ~(@1)
  .else
	ori	temp, @1
  .endif
.endmacro

oc0:	;Timer 0 output compare interrupt (PWM steps)
	push	temp
	push	temp2
	_push_w	Z
	push	R0 ;TODO: use W1~W4 instead of R0~R3 (?)
	push	R1
	push	R2
	push	R3
	in	sreg_backup, SREG
	
	;decrement row
.if (OUT_INVERT_ROWS)
	sbi	PORTE, ROW_DATA
.else
	cbi	PORTE, ROW_DATA
.endif
	dec	mux
	brne	oc0_rowinc_end
	ldi	mux, 20
.if (OUT_INVERT_ROWS)
	cbi	PORTE, ROW_DATA
.else
	sbi	PORTE, ROW_DATA
.endif
	;increment PWM value ('pwm' register stores pwm value in high nibble!)
	subi	pwm, -0x30 ;PWM value sequence: 3, 6, 1, 4, 7, 2, 5, (0)
	andi	pwm, 0x70
	brne	oc0_pwminc_end ;zero: end of PWM sequence
	ldi	pwm, 0x30 ;start with first value of PWM sequence
	tick_100hz
oc0_pwminc_end:
	;load start address
	activeframe Z ;max. 6 cycles
	_addi_w	Z, 520/2
	_sts_w	RAM_MuxAddress, Z
oc0_rowinc_end:
	
	;determine next interval
	mov	temp, pwm
	swap	temp
	_ldi_w	Z, (FLASH_OFFSET + pwm_duration) * 2
	add	ZL, temp
	adc	ZH, zero
	lpm	temp, Z
	out	OCR0, temp
	
	;load framebuffer address for current row
	_lds_w	Z, RAM_MuxAddress
	
	;set LEDs 26 to 21
	clr	R0 ;bits 0 and 1 = '0'
.if (OUT_INVERT_COLUMNS == 0)
	dec	R0 ;bits 0 and 1 = '1'  =>  '0' after complement
.endif
	_set_led_A ;7 cycles each
	_set_led_A
	_set_led_A
	
	;set LEDs 20 to 13
	_set_led_B ;7 cycles each
	_set_led_B
	_set_led_B
	_set_led_B
	
	;set LEDs 12 to 5
	_set_led_C ;7 cycles each
	_set_led_C
	_set_led_C
	_set_led_C
	
	;LEDs off
	in	temp, PORTB
.if (OUT_INVERT_COLUMNS)
	ldi	temp2, 0xFF
	mov	R3, temp2
	ldi	temp2, 0xFC
	ori	temp, 0x0F
.else
	com	R0
	com	R1
	com	R2
	clr	R3
	ldi	temp2, 0x00
	andi	temp, 0xF0
.endif
	out	PORTD, temp2
	out	PORTC, R3
	out	PORTA, R3
	out	PORTB, temp
	
	;set LEDs 4 to 1 [delay until transistors have turned off]
	_set_led_D 0x01, 0x02 ;9 cycles each
	_set_led_D 0x04, 0x08
	
	;select next row
	sbi	PORTE, ROW_CLOCK
	cbi	PORTE, ROW_CLOCK
	
	;delay until FETs have switched
	_sts_w	RAM_MuxAddress, Z ;store framebuffer address for next row
	
	;LEDs on with new data
	out	PORTD, R0
	out	PORTC, R1
	out	PORTA, R2
	out	PORTB, temp
	
	;return
	out	SREG, sreg_backup
	pop	R3
	pop	R2
	pop	R1
	pop	R0
	_pop_w	Z
	pop	temp2
	pop	temp
	reti

;===============================================================================

.macro init_output
	
	;initialize registers
	ldi	mux, 1 ;last row in sequence
	ldi	pwm, 0x50 ;last PWM value in sequence

	;initialize ports
	ldi	temp, 0xFF
	out	DDRA, temp
	out	DDRC, temp

  .if (OUT_INVERT_COLUMNS)
	ldi	temp, 0xFF
  .else
	ldi	temp, 0x00
  .endif
	out	PORTA, temp
	out	PORTC, temp
	
	ldi	temp, 0xBF
	out	DDRB, temp
	ldi	temp, 0xFE
	out	DDRD, temp
	
  .if (OUT_INVERT_COLUMNS)
	ldi	temp, 0x7F ;SPI: all high except SCK
	ldi	temp2, 0xFD ;enable pull-up for UART Rx
  .else
	ldi	temp, 0x70 ;SPI: all high except SCK
	ldi	temp2, 0x01 ;enable pull-up for UART Rx
  .endif
	out	PORTB, temp
	out	PORTD, temp2
	
	ldi	temp, 0x06
	out	DDRE, temp
	ldi	temp, 0x01
	out	PORTE, temp
	
	;initialize data in row shift register (all rows off)
  .if (OUT_INVERT_ROWS)
	sbi	PORTE, ROW_DATA
  .endif
	ldi	temp, 20
init_shift_loop:
	sbi	PORTE, ROW_CLOCK
	cbi	PORTE, ROW_CLOCK
	dec	temp
	brne	init_shift_loop

.endmacro
