;Output driver for monochrome 144-pixel (BlinkenLights) projects with 1/8-Mux
;  using ATmega128
;Supported projects:
;  * BlinkenBadge
;Supported controllers: Mega128
;
;PORTS
;  PA0-7 = [unused]
;  PB0-7 = LED Col 1..8 (MSB right)
;  PC0   = optional SD Card #CS
;  PC1   = optional SD Card MOSI
;  PC2   = optional SD Card MISO
;  PC3   = optional SD Card SCK
;  PC4-7 = LED Row 5..8 (MSB bottom)
;  PD0-7 = LED Col 11.18 (MSB right)
;  PE0/1 = RS232 Rx/Tx
;  PE2-5 = LED Row 1..4 (MSB bottom)
;  PE6/7 = [unused]
;  PF0-7 = [unused]
;  PG0-2 = [unused]
;  PG3/4 = LED Row 9/10 (MSB right)
;
;Frame Timing:
;  Controller frequency : 14.7456 MHz
;  Timer prescaler      : 256
;  PWM "steps" per row  : 72
;  Multiplexing         : 8 rows
;    => 14.7456 MHz / 256 / 72 / 8 = 100 Hz (error: 0 %)
;
;Example PWM durations (for 144-pixel projects except BlinkstroemAdvanced):
;0, 1, 3, 6, 11, 21, 39, 72  - exponential, base 1.8
;0, 2, 5, 8, 12, 21, 39, 72  - equivalent to old optimized safe exponential
;0, 1, 6, 13, 24, 37, 53, 72 - Gamma 2.0 (y = x^2 * 72/49, x = [0..7])
;0, 2, 6, 13, 24, 37, 53, 72 - Safe Gamma 2.0 (delta >= 2)
;0, 1, 3, 9, 18, 31, 49, 72  - Gamma 2.5
;absolute minimum step between two values:    1
;recommended minimum step between two values: 2 ("safe")

.equ	PWMVAL_0 = 0 ;don't change this value
.equ	PWMVAL_1 = 2
.equ	PWMVAL_2 = 5
.equ	PWMVAL_3 = 8
.equ	PWMVAL_4 = 12
.equ	PWMVAL_5 = 21
.equ	PWMVAL_6 = 39
.equ	PWMVAL_7 = 72 ;don't change this value (timing depends on it)

;===============================================================================

;SD card pins
.equ	USE_HARDWARE_SPI = 0
.equ	SD_PORT = PORTC
.equ	SD_PIN = PINC
.equ	SD_CS = 0
.equ	SD_MOSI = 1
.equ	SD_MISO = 2
.equ	SD_CK = 3
.equ	SDCARD_HAS_DETECT_PIN = 0

;other settings
.equ	OUT_T0_DIV = 0x06 ;Clk/256 (value not compatible with other MCUs)
.equ	DISALLOW_UART = 0
.equ	DISALLOW_UART_TX = 0

;===============================================================================

.macro _set_led_1 ;LEDs 1 to 8
	ld	temp, Z+
	swap	temp
	cp	temp, pwm ;carry set if pwm > temp
	ror	R0
	swap	temp
	cp	temp, pwm ;carry set if pwm > temp
	ror	R0
.endmacro
	
.macro _set_led_2 ;LEDs 9 and 10
	ld	temp, Z+
	cp	temp, pwm
	brlo	PC+2
  .if (OUT_INVERT_COLUMNS)
	andi	temp2, LOW(~(@1)) ;'LOW' prevents assembler warning
  .else
	ori	temp2, @1
  .endif
	swap	temp
	cp	temp, pwm
	brlo	PC+2
  .if (OUT_INVERT_COLUMNS)
	andi	temp2, LOW(~(@0)) ;'LOW' prevents assembler warning
  .else
	ori	temp2, @0
  .endif
.endmacro

.macro _set_led_3 ;LEDs 11 to 18
	ld	temp, Z+
	swap	temp
	cp	temp, pwm ;carry set if pwm > temp
	ror	R1
	swap	temp
	cp	temp, pwm ;carry set if pwm > temp
	ror	R1
.endmacro

;===============================================================================
	
oc0:	;Timer 0 output compare interrupt (PWM steps)
	in	sreg_backup, SREG
	push	temp
	push	temp2
	_push_w	Z
	push	R0
	push	R1
	
	;increment row
	lsl	mux
	brne	oc0_rowinc_end
	ldi	mux, 0x01
	;increment PWM value ('pwm' register stores pwm value in high nibble!)
	subi	pwm, -0x30 ;PWM value sequence: 3, 6, 1, 4, 7, 2, 5, (0)
	andi	pwm, 0x70
	brne	oc0_pwminc_end ;zero: end of PWM sequence
	ldi	pwm, 0x30 ;start with first value of PWM sequence
	tick_100hz
oc0_pwminc_end:
	;load start address
	activeframe Z ;max. 6 cycles
	_sts_w	RAM_MuxAddress, Z
oc0_rowinc_end:
	
	;determine next interval
	mov	temp, pwm
	swap	temp
	_ldi_w	Z, (FLASH_OFFSET + pwm_duration) * 2
	add	ZL, temp
	adc	ZH, zero
	lpm	temp, Z
	out	OCR0, temp
	
	;load framebuffer address for current row
	_lds_w	Z, RAM_MuxAddress
	
	;set LEDs 1 to 8
	_set_led_1 ;8 cycles each
	_set_led_1
	_set_led_1
	_set_led_1
	
	;set LEDs 9, 10
	_in	temp2, PORTG
.if (OUT_INVERT_COLUMNS)
	ori	temp2, 0x18
.else
	andi	temp2, 0xE7
.endif
	_set_led_2 0x08, 0x10 ;9 cycles
	
	;set LEDs 11 to 18
	_set_led_3 ;8 cycles each
	_set_led_3
	_set_led_3
	_set_led_3
	
.if (OUT_INVERT_COLUMNS == 0)
	com	R0
	com	R1
.endif
	
	;old row off
	in	temp, PORTC
.if (OUT_INVERT_ROWS)
	ori	temp, 0xF0
.else
	andi	temp, 0x0F
.endif
	out	PORTC, temp
	in	temp, PORTE
.if (OUT_INVERT_ROWS)
	ori	temp, 0x3C
.else
	andi	temp, 0xC3
.endif
	out	PORTE, temp
	
	;delay until transistor is off
	_sts_w	RAM_MuxAddress, Z ;store framebuffer address for next row
	
	;output new LED data
	out	PORTB, R0
	out	PORTD, R1
	_out	PORTG, temp2
	
	;more delay
	pop	R1
	pop	R0
	_pop_w	Z
	
	;new row on
	in	temp, PORTC
	mov	temp2, mux
	andi	temp2, 0xF0
.if (OUT_INVERT_ROWS)
	com	temp2
	and	temp, temp2
.else
	or	temp, temp2
.endif
	out	PORTC, temp
	_in	temp, PORTE
	mov	temp2, mux
	lsl	temp2
	lsl	temp2
	andi	temp2, 0x3F
.if (OUT_INVERT_ROWS)
	com	temp2
	and	temp, temp2
.else
	or	temp, temp2
.endif
	_out	PORTE, temp
	
	;return
	pop	temp2
	pop	temp
	out	SREG, sreg_backup
	reti

;===============================================================================

.macro init_output
	
	;initialize registers
	ldi	mux, 0x00 ;no row
	ldi	pwm, 0x50 ;last PWM value in sequence

	;initialize ports
	ldi	temp, 0x00
	out	DDRA, temp
	_out	DDRF, temp
	ldi	temp, 0xFF
	out	PORTA, temp ;unused port: all pins = input with pull-up
	_out	PORTF, temp ;unused port: all pins = input with pull-up
	out	DDRB, temp
	out	DDRD, temp
  .if (OUT_INVERT_COLUMNS)
	ldi	temp, 0xFF
  .else
	ldi	temp, 0x00
  .endif
	out	PORTB, temp
	out	PORTD, temp
	ldi	temp, 0xF0 | 1<<SD_CS | 1<<SD_MOSI | 1<<SD_CK
	out	DDRC, temp
  .if (OUT_INVERT_ROWS)
	ldi	temp, 0xF0 | 1<<SD_CS | 1<<SD_MOSI | 1<<SD_MISO
  .else
	ldi	temp, 0x00 | 1<<SD_CS | 1<<SD_MOSI | 1<<SD_MISO
  .endif
	out	PORTC, temp
	ldi	temp, 0x3E
	out	DDRE, temp
  .if (OUT_INVERT_ROWS)
	ldi	temp, 0xFD ;pull-ups on unused pins & RxD
  .else
	ldi	temp, 0xC1 ;pull-ups on unused pins & RxD
  .endif
	out	PORTE, temp
	ldi	temp, 0x18
	_out	DDRG, temp
  .if (OUT_INVERT_COLUMNS)
	ldi	temp, 0x1F ;pull-ups on unused pins
  .else
	ldi	temp, 0x07 ;pull-ups on unused pins
  .endif
	_out	PORTG, temp

.endmacro
