;Output driver for stream splitter
;The source frames are split horizontally into multiple output frames which
;are then sent to up to 4 software UART outputs (in addition to the non-split
;or debugging output available on the hardware UART). HEIGHT must be set to a
;multiple of the number of split outputs (in config.inc).
;Supported controllers: Mega644
;
;PORTS (ATmega644):
;  PA0-7 = [unused]
;  PB0-2 = [unused]
;  PB3   = SD-Card Detect Switch (low = card is present)
;  PB4   = SD-Card #CS (voltage divider 1k/2.2k)
;  PB5   = SD-Card D_in (MOSI) (voltage divider 1k/2.2k)
;  PB6   = SD-Card D_out (MISO) (series resistor 1k)
;  PB7   = SD-Card Clock (voltage divider 1k/2.2k)
;  PC0-7 = [unused]
;  PD0/1 = RS232 Rx/Tx (full stream input, debugging or full stream output)
;  PD2-3 = [unused]
;  PD4   = RS232 Tx (split stream output 4)
;  PD5   = RS232 Tx (split stream output 3)
;  PD6   = RS232 Tx (split stream output 2)
;  PD7   = RS232 Tx (split stream output 1)
;
;Frame Timing:
;  Controller frequency : 14.7456 MHz
;  Timer prescaler      : 1024
;  Interval             : 144
;    => 14.7456 MHz / 1024 / 144 = 100 Hz (error: 0 %)

;===============================================================================

;SD card pins
.equ	USE_HARDWARE_SPI = 1
.equ	SD_PORT = PORTB
.equ	SD_CS = 4
.equ	SDCARD_HAS_DETECT_PIN = 1
.equ	SD_DETECT_PIN = PINB
.equ	SD_DETECT = 3 ;Card Detect switch pin on PORTB

;other settings
.equ	OUT_T0_DIV = 0x05 ;Clk/1024
.equ	DISALLOW_UART = 0
.equ	DISALLOW_UART_TX = 0

;===============================================================================

.equ	SPLIT_OUTPUTS = SPLIT_V * SPLIT_H

;unused, but referenced in blinkenplus.asm => values don't matter
.equ	PWMVAL_0 = 0
.equ	PWMVAL_1 = 18
.equ	PWMVAL_2 = 36
.equ	PWMVAL_3 = 54
.equ	PWMVAL_4 = 72
.equ	PWMVAL_5 = 90
.equ	PWMVAL_6 = 108
.equ	PWMVAL_7 = 144

;===============================================================================

oc0:	;Timer 0 output compare interrupt (100 Hz frame interrupt)
	push	temp2
	in	sreg_backup, SREG
	
	;set TOP value for next timer cycle
	ldi	temp2, 144-1
	out	OCR0A, temp2
	
.if (STREAM_INPUT)
	;decrement RS232 timeout counter
	lds	temp2, RAM_Timeout_RS232
	tst	temp2
	breq	oc0_timeout_end
	dec	temp2
	sts	RAM_Timeout_RS232, temp2
oc0_timeout_end:
.endif
	
	;TODO: merge following code block with "tick_100hz" macro in
	;      blinkenplus.asm
	;decrement frame display duration if > 0
	_tst_w	time
	breq	oc0_time_zero
	_subi_w	time, 1
	brne	oc0_time_end
oc0_time_zero:
	sbrs	flags, fNewFrame
	rjmp	oc0_time_end
	_in	temp2, TIMSK2
	_sbic	TIMSK2, OCIE2A
	rjmp	oc0_time_end ;transmission of last frame still in progress
	;NEW FRAME:
	;swap active/inactive frame, clear fNewFrame:
	ldi	temp2, 1<<fActiveFrame | 1<<fNewFrame
	eor	flags, temp2
	_lds_w	time, RAM_Duration ;load duration for new frame
	;start transmissions on split-output UARTs
	sts	RAM_MuxAddress, zero
	sts	RAM_MuxAddress+1, zero
	ldi	line, WIDTH * CHANNELS / SPLIT_V + 12 ;add 12 bytes for header
oc0_t2enable_wait:
	_in	temp2, TCNT2
	cpi	temp2, 100 ;don't enable oc2 IRQ 15 cycles or less before TOP
	brsh	oc0_t2enable_wait
	ldi	temp2, 1<<OCIE2A ;enable compare match A interrupt (start Tx)
	_out	TIMSK2, temp2
	;enable UDR empty interrupt (start transmitting on main UART)
.if (STREAM_OUTPUT)
  .if (UCSRB < 0x20)
	sbi	UCSRB, UDRIE
  .else
	_in	temp, UCSRB
	ori	temp, 1<<UDRIE
	_out	UCSRB, temp
  .endif
.endif
	
oc0_time_end:
	;return
	out	SREG, sreg_backup
	pop	temp2
	reti

;===============================================================================

.dseg
	;RAM buffer for bytes currently transmitted
.if (SPLIT_OUTPUTS <= 2)
	RAM_Split_Bytes: .byte 2
.elif (SPLIT_OUTPUTS <= 4)
	RAM_Split_Bytes: .byte 4
.else
	.error "Too many split outputs set (max. 4). Reduce SPLIT_H or SPLIT_V."
.endif
.cseg

;--------------------

oc2_pos:
.org FLASH_OFFSET + OC2Aaddr
	;interrupt vector for Timer 2 output compare
	rjmp	oc2
.org oc2_pos

;--------------------

split_header:
	;MCUF header for split frames
	.db 0x23, 0x54, 0x26, 0x66 ;magic
	.db 0x00, HEIGHT / SPLIT_H, 0x00, WIDTH / SPLIT_V ;height, width
	.db 0x00, CHANNELS, 0x00, 0x07 ;channels, maxval 0x07 (8 grayscales)

;--------------------

oc2:
	;software UART (Tx only) interrupt
	push	temp
	_push_w	Z
	in	sreg_backup, SREG
	
	cpi	bit, 0
	_breq	oc2_startbit
	cpi	bit, 9
	brsh	oc2_stopbit
	
	;data bit
	ldi	temp, 1<<WGM21 | 1<<COM2A1 | 1<<COM2B1 ;clear
	_lds_w	Z, RAM_Split_Bytes
	sbrc	ZL, 0
	ori	temp, 1<<COM2A0 ;set
	sbrc	ZH, 0
	ori	temp, 1<<COM2B0 ;set
	_out	TCCR2A, temp
	lsr	ZL
	lsr	ZH
	_sts_w	RAM_Split_Bytes, Z
.if (SPLIT_OUTPUTS >= 3)
	ldi	temp, 1<<COM1A1 | 1<<COM1B1 ;clear
	_lds_w	Z, RAM_Split_Bytes+2
	sbrc	ZL, 0
	ori	temp, 1<<COM1A0 ;set
	sbrc	ZH, 0
	ori	temp, 1<<COM1B0 ;set
	_out	TCCR1A, temp
	lsr	ZL
	lsr	ZH
	_sts_w	RAM_Split_Bytes+2, Z
.endif
	inc	bit
	rjmp	oc2_end

oc2_stopbit:
	;stop bit
	ldi	temp, 1<<WGM21 | 1<<COM2A1|1<<COM2A0 | 1<<COM2B1|1<<COM2B0 ;set
	_out	TCCR2A, temp
.if (SPLIT_OUTPUTS >= 3)
	ldi	temp, 1<<COM1A1 | 1<<COM1A0 | 1<<COM1B1 | 1<<COM1B0 ;set
	_out	TCCR1A, temp
.endif
	ldi	bit, 0
	;increment RAM address
	_lds_w	Z, RAM_MuxAddress
	adiw	ZH:ZL, 1
.if (SPLIT_V >= 2)
	dec	line
	brne	oc2_vsplit_end
	ldi	line, WIDTH * CHANNELS / SPLIT_V
	adiw	ZH:ZL, WIDTH * CHANNELS / SPLIT_V * (SPLIT_V - 1)
oc2_vsplit_end:
.endif
	_sts_w	RAM_MuxAddress, Z
	;check if transmission complete (12 bytes header + frame data)
	ldi	temp, HIGH(12 + N_LEDS / SPLIT_H)
	cpi	ZL, LOW(12 + N_LEDS / SPLIT_H)
	cpc	ZH, temp
	_brlo	oc2_end
	;transmission complete
	ldi	temp, 0 ;disable interrupt
	_out	TIMSK2, temp
	rjmp	oc2_end
	
oc2_startbit:
	;start bit
	ldi	temp, 1<<WGM21 | 1<<COM2A1 | 1<<COM2B1
	_out	TCCR2A, temp
.if (SPLIT_OUTPUTS >= 3)
	ldi	temp, 1<<COM1A1 | 1<<COM1B1
	_out	TCCR1A, temp
.endif
	;load bytes to send
	ldi	bit, 1
	_lds_w	Z, RAM_MuxAddress
	cpi	ZL, 12 ;first 12 bytes are header
	cpc	ZH, zero
	brlo	oc2_header
	;load data bytes
	sbiw	ZH:ZL, 12
	bst	ZL, 0
	_lsr_w	Z
	_addi_w	Z, RAM_Frame0
	sbrs	flags, fActiveFrame
	rjmp	oc2_activeframe_end
	_addi_w	Z, (RAM_Frame1 - RAM_Frame0)
oc2_activeframe_end:
	ld	temp, Z
	brtc	PC+2
	swap	temp
	andi	temp, 0x07
	sts	RAM_Split_Bytes, temp
.if (SPLIT_V >= 2)
	_addi_w	Z, WIDTH/2 * CHANNELS / SPLIT_V
.else
	_addi_w	Z, N_LEDS/2 / SPLIT_H
.endif
	ld	temp, Z
	brtc	PC+2
	swap	temp
	andi	temp, 0x07
	sts	RAM_Split_Bytes+1, temp
.if (SPLIT_OUTPUTS >= 3)
  .if (SPLIT_V >= 3)
	_addi_w	Z, WIDTH/2  * CHANNELS / SPLIT_V
  .elif (SPLIT_V == 2)
	_addi_w	Z, (N_LEDS/2 / SPLIT_H) - (WIDTH/2  * CHANNELS / SPLIT_V)
  .else
	_addi_w	Z, N_LEDS/2 / SPLIT_H
  .endif
	ld	temp, Z
	brtc	PC+2
	swap	temp
	andi	temp, 0x07
	sts	RAM_Split_Bytes+2, temp
  .if (SPLIT_V >= 2)
	_addi_w	Z, WIDTH/2  * CHANNELS / SPLIT_V
  .else
	_addi_w	Z, N_LEDS/2 / SPLIT_H
  .endif
	ld	temp, Z
	brtc	PC+2
	swap	temp
	andi	temp, 0x07
	sts	RAM_Split_Bytes+3, temp
.endif

oc2_end:
	out	SREG, sreg_backup
	_pop_w	Z
	pop	temp
	reti
	
oc2_header:
	;load header bytes
	_addi_w	Z, (FLASH_OFFSET + split_header) * 2
	lpm	temp, Z
	sts	RAM_Split_Bytes, temp
	sts	RAM_Split_Bytes+1, temp
.if (SPLIT_OUTPUTS >= 3)
	sts	RAM_Split_Bytes+2, temp
	sts	RAM_Split_Bytes+3, temp
.endif
	rjmp	oc2_end

;===============================================================================

.macro init_output
	
	;initialize ports
	;set unused ports A and C to inputs with pull-ups
	ldi	temp, 0x00
	out	DDRA, temp
	out	DDRC, temp
	ldi	temp, 0xFF
	out	PORTA, temp
	out	PORTC, temp
	;PORTB: SD-Card (unused pins: inputs with pull-ups)
	ldi	temp, 0xB0
	out	DDRB, temp
	ldi	temp, 0x7F ;SPI: all high except SCK, pull-up for MISO & detect
	out	PORTB, temp
	;PORTD: UARTs (unused pins: inputs with pull-ups)
	ldi	temp, 0xF2
	out	DDRD, temp
	ldi	temp, 0xFF ;pull-up for RxD
	out	PORTD, temp
	
	;initialize registers
	ldi	bit, 0
	ldi	line, 0
	
	;initialize Timer 2 for UART outputs #1 and #2:
	;CTC mode (TOP = OCR2A), compare match on 127
	ldi	temp, 127 ;14.7456 MHz / 128 = 115200 Baud
	_out	OCR2A, temp
	_out	OCR2B, temp
	ldi	temp, 1<<WGM21 | 1<<COM2A1|1<<COM2A0 | 1<<COM2B1|1<<COM2B0 ;set
	_out	TCCR2A, temp
	ldi	temp, 0x01 ;Clk/1
	_out	TCCR2B, temp
	
  .if (SPLIT_OUTPUTS >= 3)
	;initialize Timer 1 for UART outputs #3 and #4:
	;CTC mode (TOP = OCR1A), compare match on 127
	ldi	temp, LOW(127) ;14.7456 MHz / 128 = 115200 Baud
	ldi	temp2, HIGH(127)
	_out	OCR1AH, temp2
	_out	OCR1AL, temp
	_out	OCR1BH, temp2
	_out	OCR1BL, temp
	ldi	temp, 1<<COM1A1 | 1<<COM1A0 | 1<<COM1B1 | 1<<COM1B0 ;set
	_out	TCCR1A, temp
	ldi	temp, 1<<WGM12 | 0x01 ;Clk/1
	_out	TCCR1B, temp
	
	;sync timers
	ldi	temp, 0
	_out	TCNT1H, temp
	_out	TCNT1L, temp
	ldi	temp, 3      ;ldi = 1 cycle
	_out	TCNT2, temp  ;sts = 2 cycles
  .endif
	
.endmacro
