;Output driver for RGB colour 520-pixel (ARCADE) projects with 1/20-Mux
;Supported projects:
;  * ArcadeMicro Colour
;Supported controllers: Mega644A, -P or -PA (not Mega644!)
;
;PORTS (ATmega644):
;  PA0-7 = [unused]
;  PB0-2 = [unused]
;  PB3   = SD-Card Detect Switch (low = card is present)
;  PB4   = SD-Card #CS (voltage divider 1k/2.2k)
;  PB5   = SD-Card D_in (MOSI) (voltage divider 1k/2.2k)
;  PB6   = SD-Card D_out (MISO) (resistor 1k)
;  PB7   = SD-Card Clock (voltage divider 1k/2.2k)
;  PC0-7 = [unused]
;  PD0/1 = RS232 Rx/Tx
;  PD2   = LED Column Shift Register Output Enable (active low)
;  PD3   = LED Column Shift Register Data
;  PD4   = LED Column Shift Register Clock
;  PD5   = LED Column Shift Register Latch
;  PD6   = LED Row Shift Register Data
;  PD7   = LED Row Shift Register Clock
;
;Shift Registers:
;  Columns:
;    * 80 bits total (78 bits used)
;    * two dummy bits shifted first (appear on Q80, Q79)
;    * followed by rightmost (column 26) LED blue data (appears on Q78)
;    * leftmost (column 1) LED red data shifted last (appears on Q1)
;  Rows:
;    * 24 bits total (20 bits used)
;    * Q1 = row 20 = bottom ... Q20 = row 1 = top (scanning from bottom to top)
;    * Q21~Q24 unused
;
;Frame Timing:
;  Controller frequency : 14.7456 MHz
;  Timer prescaler      : 64
;  PWM "steps" per row  : 115
;  Multiplexing         : 20 rows
;    => 14.7456 MHz / 64 / 115 / 20 = ~100.174 Hz (error: ~0.174 %)
;
;Example PWM durations:
;0, 7, 14, 21, 28, 35, 62, 115 - similar to old optimized exponential
;0, 7, 14, 21, 38, 59, 84, 115 - Gamma 2.0
;absolute minimum step between two values: 7 (= 448 cycles)
.equ	PWMVAL_0 = 0 ;don't change this value
.equ	PWMVAL_1 = 7
.equ	PWMVAL_2 = 14
.equ	PWMVAL_3 = 21
.equ	PWMVAL_4 = 28
.equ	PWMVAL_5 = 35
.equ	PWMVAL_6 = 62
.equ	PWMVAL_7 = 115 ;don't change this value (timing depends on it)

;TODO: use 18.432 MHz for better grayscales? (144 pwmsteps, 0% error)


;===============================================================================

;SD card pins
.equ	USE_HARDWARE_SPI = 1
.equ	SD_PORT = PORTB
.equ	SD_CS = 4
.equ	SDCARD_HAS_DETECT_PIN = 1
.equ	SD_DETECT_PIN = PINB
.equ	SD_DETECT = 3 ;Card Detect switch pin

;shift register pins (PORTD)
.equ	COL_ENABLE = 2
.equ	COL_LATCH = 5
.equ	ROW_DATA = 6
.equ	ROW_CLOCK = 7

;other settings
.equ	OUT_T0_DIV = 0x03 ;Clk/64
.equ	DISALLOW_UART = 0
.equ	DISALLOW_UART_TX = 0

;===============================================================================

.if ((MCU != MCU_MEGA644A) && (MCU != MCU_MEGA644P))
	.error "Output module for ARCADE RGB only works with ATmega644A/P/PA!"
.endif

.macro shift_led_byte ;29/30 cycles ('out') or 30/31 cycles ('sts')
	set_2_leds
	set_2_leds
	set_2_leds
	set_2_leds
  .if (OUT_INVERT_COLUMNS == 0)
	com	temp2
  .endif
	_out	UDR1, temp2
.endmacro

.macro set_2_leds ;7 cycles
	ld	temp, -Z
	cp	temp, pwm
	rol	temp2
	swap	temp
	cp	temp, pwm
	rol	temp2
.endmacro

;--------------------

oc0:	;Timer 0 output compare interrupt (PWM steps)
	push	temp
	push	temp2
	_push_w	Z
	in	sreg_backup, SREG
	
	;decrement row
.if (OUT_INVERT_ROWS)
	sbi	PORTD, ROW_DATA
.else
	cbi	PORTD, ROW_DATA
.endif
	dec	mux
	brne	oc0_rowinc_end
	ldi	mux, HEIGHT
.if (OUT_INVERT_ROWS)
	cbi	PORTD, ROW_DATA
.else
	sbi	PORTD, ROW_DATA
.endif
	;increment PWM value ('pwm' register stores pwm value in high nibble!)
	subi	pwm, -0x30 ;PWM value sequence: 3, 6, 1, 4, 7, 2, 5, (0)
	andi	pwm, 0x70
	brne	oc0_pwminc_end
	;zero: end of PWM sequence
	ldi	pwm, 0x30 ;start with first value of PWM sequence
	tick_100hz ;max. 23 cycles
oc0_pwminc_end:
	;load start address: end of bottom row
	activeframe Z ;max. 6 cycles
	_addi_w	Z, N_LEDS / 2
	_sts_w	RAM_MuxAddress, Z
oc0_rowinc_end:
	
	;determine next interval
	mov	temp, pwm
	swap	temp
	_ldi_w	Z, (FLASH_OFFSET + pwm_duration) * 2
	add	ZL, temp
	adc	ZH, zero
	lpm	temp, Z
	out	OCR0, temp
	
	;load framebuffer address for current row
	_lds_w	Z, RAM_MuxAddress
	
	;69 cycles to this point (not including jump to 'oc0')
	
	ldi	temp2, 0xFF
	set_2_leds ;columns 78, 77
	set_2_leds ;columns 76, 75
	set_2_leds ;columns 74, 73
.if (OUT_INVERT_COLUMNS == 0)
	com	temp2
.endif
	_out	UDR1, temp2
	shift_led_byte ;columns 72 ~ 65
	shift_led_byte ;columns 64 ~ 57
	shift_led_byte ;columns 56 ~ 49
	shift_led_byte ;columns 48 ~ 41
	shift_led_byte ;columns 40 ~ 33
	shift_led_byte ;columns 32 ~ 25
	shift_led_byte ;columns 24 ~ 17
.if (OUT_AMC_REV1_CORRECTION)
	set_2_leds ;columns 16, 15
	set_2_leds ;columns 14, 13
	;columns 11 and 12 are swapped
	ld	temp, -Z
	swap	temp
	cp	temp, pwm
	rol	temp2
	swap	temp
	cp	temp, pwm
	rol	temp2
	set_2_leds ;columns 10,  9
	
  .if (OUT_INVERT_COLUMNS == 0)
	com	temp2
  .endif
	_out	UDR1, temp2
.else
	shift_led_byte ;columns 16 ~  9
.endif
	shift_led_byte ;columns  8 ~  1
	
	;max. 69 + 304 = 373 cycles to this point (not including jump to 'oc0')
	
	;store framebuffer address for next row
	_sts_w	RAM_MuxAddress, Z
	
	;restore registers (delay to wait for last SPI transfer)
	out	SREG, sreg_backup
	_pop_w	Z
	pop	temp2

	;update LEDs
	sbi	PORTD, COL_ENABLE ;LEDs off
	sbi	PORTD, ROW_CLOCK ;advance to next row
	cbi	PORTD, ROW_CLOCK
	sbi	PORTD, COL_LATCH ;latch new column data
	cbi	PORTD, COL_LATCH ;  (>= 19 cycles after last 'out' incl. 'sbi')
	pop	temp
	cbi	PORTD, COL_ENABLE ;LEDs on (with new data)
	
	;return
	reti
	
	;max. 69 + 304 + 29 = 402 cycles (not including jump to 'oc0')

;===============================================================================

.macro init_output
	
	;initialize registers
	ldi	mux, 1 ;last row in sequence
	ldi	pwm, 0x50 ;last PWM value in sequence

	;initialize ports
	;unused PORTA and PORTC: inputs with pull-ups
	ldi	temp, 0x00
	out	DDRA, temp
	out	DDRC, temp
	ldi	temp, 0xFF
	out	PORTA, temp
	out	PORTC, temp
	;PORTB: SD-Card & unused
	ldi	temp, 0xB0 ;outputs: SCK, MOSI, CS
	out	DDRB, temp
	ldi	temp, 0x4F ;pull-ups for MISO and unused pins
	out	PORTB, temp
	;PORTD: UART & shift registers: pull-up for RxD
	ldi	temp, 0xFE
	out	DDRD, temp
	ldi	temp, 1<<COL_ENABLE | 0x03 ;COL_ENABLE is active low
	out	PORTD, temp
	
	;initialize data in row shift register (all rows off)
  .if (OUT_INVERT_ROWS)
	sbi	PORTD, ROW_DATA
  .endif
	ldi	temp, 20
init_shift_loop:
	sbi	PORTD, ROW_CLOCK
	cbi	PORTD, ROW_CLOCK
	dec	temp
	brne	init_shift_loop
	
	;initialize USART1 for SPI master mode (column shift register)
	;(Tx only, MSB first, setup on falling edge, sample on rising edge)
	ldi	temp, 1<<UMSEL11 | 1<<UMSEL10
	_out	UCSR1C, temp
	ldi	temp, 1<<TXEN
	_out	UCSR1B, temp
	ldi	temp, 0
	_out	UBRR1H, temp
	_out	UBRR1L, temp
	
.endmacro
