;Output driver for dual-colour 160-pixel (20x8) projects with 1/4-Mux
;physical pixel arrangement: 20x8, logical pixel arrangement: 40x4
;Supported projects:
;  * 20x8 bi-colour matrix display
;Supported controllers: Mega16, Mega32
;
;PORTS (ATmega16, 20x8 bi-colour display):
;  PA0-7 = LED Col data (LSB left, to latches, Col 0 = red, Col 1 = green, ...)
;  PB0-3 = LED Row 1..4 (row 1 = row 5, row 2 = row 6, ...)
;  PB4   = SD-Card #CS (voltage divider 1k/2.2k)
;  PB5   = SD-Card D_in (MOSI) (voltage divider 1k/2.2k)
;  PB6   = SD-Card D_out (MISO) (series resistor 1k)
;  PB7   = SD-Card Clock (voltage divider 1k/2.2k)
;  PC0-7 = Strobe outputs for Col groups 1-8, 9-16, ..., 57-64
;  PD0/1 = RS232 Rx/Tx
;  PD2   = SD-Card Detect Switch (low = card is present)
;  PD3-5 = [unused]
;  PD6/7 = Strobe outputs for Col groups 65-72 and 73-80
;
;Frame Timing:
;  Controller frequency : 14.7456 MHz
;  Timer prescaler      : 256
;  PWM "steps" per row  : 144
;  Multiplexing         : 4 rows
;    => 14.7456 MHz / 256 / 144 / 4 = 100 Hz (error: 0 %)
;
;Example PWM durations (for 160-pixel projects):
;0, 2, 5, 12, 23, 43, 79, 144 - exponential, base 1.8
;absolute minimum step between two values: 2 TODO check this

.equ	PWMVAL_0 = 0 ;don't change this value
.equ	PWMVAL_1 = 2
.equ	PWMVAL_2 = 5
.equ	PWMVAL_3 = 12
.equ	PWMVAL_4 = 23
.equ	PWMVAL_5 = 43
.equ	PWMVAL_6 = 79
.equ	PWMVAL_7 = 144 ;don't change this value (timing depends on it)

;===============================================================================

;SD card pins
.equ	USE_HARDWARE_SPI = 1
.equ	SD_PORT = PORTB
.equ	SD_CS = 4
.equ	SDCARD_HAS_DETECT_PIN = 1
.equ	SD_DETECT_PIN = PIND
.equ	SD_DETECT = 2 ;Card Detect switch pin on PORTD

;other settings
.equ	OUT_T0_DIV = 0x04 ;Clk/256
.equ	DISALLOW_UART = 0
.equ	DISALLOW_UART_TX = 0

;===============================================================================

.macro _set_led ;8 cycles
	ld	temp, Z+
	swap	temp
	cp	temp, pwm ;carry set if pwm > temp
	ror	@0
	swap	temp
	cp	temp, pwm ;carry set if pwm > temp
	ror	@0
.endmacro

.macro _set_8_leds ;32/33 cycles (inverted/non-inverted)
	_set_led @0
	_set_led @0
	_set_led @0
	_set_led @0
  .if (OUT_INVERT_COLUMNS == 0)
	com	@0
  .endif
.endmacro

;===============================================================================

oc0:	;Timer 0 output compare interrupt (PWM steps)
	push	temp
	push	temp2
	push	R0
	push	R1
	push	R2
	push	R3
	push	R4
	push	R5
	push	R6
	push	R7
	push	R8
	push	R9
	_push_w	Z
	in	sreg_backup, SREG
	
	;increment row
	lsl	mux
	brhc	oc0_rowinc_end
	ldi	mux, 0x01
	;increment PWM value ('pwm' register stores pwm value in high nibble!)
	subi	pwm, -0x30 ;PWM value sequence: 3, 6, 1, 4, 7, 2, 5, (0)
	andi	pwm, 0x70
	brne	oc0_pwminc_end ;zero: end of PWM sequence
	ldi	pwm, 0x30 ;start with first value of PWM sequence
	tick_100hz
oc0_pwminc_end:
	;load start address
	activeframe Z ;max. 6 cycles
	_sts_w	RAM_MuxAddress, Z
oc0_rowinc_end:
	
	;determine next interval
	mov	temp, pwm
	swap	temp
	_ldi_w	Z, (FLASH_OFFSET + pwm_duration) * 2
	add	ZL, temp
	adc	ZH, zero
	lpm	temp, Z
	out	OCR0, temp
	
	;load framebuffer address for current row
	_lds_w	Z, RAM_MuxAddress
	
	;update LED latches (35 cycles per latch = 350 cycles), converting from
	;physical geometry (20x8, as stored in RAM) to virtual (multiplexing)
	;geometry (40x4, i.e. 4 multiplexing rows)
	_set_8_leds R0 ;cols 1-8 (physical rows 1/2/3/4 cols 1-8)
	_set_8_leds R1 ;cols 9-16 (physical rows 1/2/3/4 cols 9-16)
	_set_8_leds R2 ;cols 17-24 (physical rows 1/2/3/4 cols 17-24)
	_set_8_leds R3 ;cols 25-32 (physical rows 1/2/3/4 cols 25-32)
	_set_8_leds R4 ;cols 33-40 (physical rows 1/2/3/4 cols 33-40)
	adiw	ZH:ZL, WIDTH/2 * CHANNELS * (HEIGHT/2 - 1)
	_set_8_leds R5 ;cols 41-48 (physical rows 5/6/7/8 cols 1-8)
	_set_8_leds R6 ;cols 49-56 (physical rows 5/6/7/8 cols 9-16)
	_set_8_leds R7 ;cols 57-64 (physical rows 5/6/7/8 cols 17-24)
	_set_8_leds R8 ;cols 65-72 (physical rows 5/6/7/8 cols 25-32)
	_set_8_leds R9 ;cols 73-80 (physical rows 5/6/7/8 cols 33-40)
	_subi_w	Z, WIDTH/2 * CHANNELS * (HEIGHT/2)
	
	;old (virtual) row off
	in	temp2, PORTB
.if (OUT_INVERT_ROWS)
	ori	temp2, 0x0F ;row outputs high, SPI outputs unchanged
.else
	andi	temp2, 0xF0 ;row outputs low, SPI outputs unchanged
.endif
	out	PORTB, temp2
	
	;delay until FET is off
	eor	temp2, mux
	_sts_w	RAM_MuxAddress, Z ;store framebuffer address for next row
	
	;output new LED data
	out	PORTA, R0 ;cols 1-8
	sbi	PORTC, 0
	out	PORTC, zero
	out	PORTA, R1 ;cols 9-16
	sbi	PORTC, 1
	out	PORTC, zero
	out	PORTA, R2 ;cols 17-24
	sbi	PORTC, 2
	out	PORTC, zero
	out	PORTA, R3 ;cols 25-32
	sbi	PORTC, 3
	out	PORTC, zero
	out	PORTA, R4 ;cols 33-40
	sbi	PORTC, 4
	out	PORTC, zero
	out	PORTA, R5 ;cols 41-48
	sbi	PORTC, 5
	out	PORTC, zero
	out	PORTA, R6 ;cols 49-56
	sbi	PORTC, 6
	out	PORTC, zero
	out	PORTA, R7 ;cols 57-64
	sbi	PORTC, 7
	out	PORTC, zero
	out	PORTA, R8 ;cols 65-72
	sbi	PORTD, 6
	cbi	PORTD, 6
	out	PORTA, R9 ;cols 73-80
	sbi	PORTD, 7
	cbi	PORTD, 7
	
	;delay until transistors have switched
	out	SREG, sreg_backup
	_pop_w	Z
	
	;new (virtual) row on
	out	PORTB, temp2

	;return
	pop	R9
	pop	R8
	pop	R7
	pop	R6
	pop	R5
	pop	R4
	pop	R3
	pop	R2
	pop	R1
	pop	R0
	pop	temp2
	pop	temp
	reti

;===============================================================================

.macro init_output
	
	;initialize registers
	ldi	mux, 0x01 ;no row
	ldi	pwm, 0x50 ;last PWM value in sequence

	;initialize ports
	ldi	temp, 0xFF
	out	DDRA, temp
	out	DDRC, temp
	ldi	temp, 0x00
	out	PORTA, temp
	out	PORTC, temp
	ldi	temp, 0xBF
	out	DDRB, temp
	;unused pins on PORTD = inputs (with pull-up)
	ldi	temp, 0xC2
	out	DDRD, temp
	
.if (OUT_INVERT_ROWS)
	ldi	temp, 0x7F ;SPI: all high except SCK, rows: all off (high)
.else
	ldi	temp, 0x70 ;SPI: all high except SCK, rows: all off (low)
.endif
	out	PORTB, temp
	;pull-ups for UART Rx, SD card detect and unused pins
	ldi	temp, 0x3F | 1<<SD_DETECT
	out	PORTD, temp
	
.endmacro
