;Output driver for monochrome 144-pixel (BlinkenLights) projects with 1/8-Mux
;Supported projects:
;  * BlinkenLEDs Plus (Prototype, Rev. 1 and Rev. 3)
;  * BlinkstroemAdvanced Plus
;  * TicTacLights
;Supported controllers: Mega16, Mega8515, Mega162, probably Mega32
;
;PORTS (ATmega8515 or ATmega16, 144 pixels, except BlinkstroemAdvanced[-Stream])
;  PA0-7 = LED Col 8..1 (MSB left) [BSA-Plus & TicTacLights: LED Col 1..8
;                                  (MSB right)]
;  PB0-7 = LED Row 1..8 (MSB bottom)
;  PC0-7 = LED Col 16..9 (MSB left)
;  PD0/1 = RS232 Rx/Tx
;  PD2   = SD-Card #CS (voltage divider 1k/2.2k)
;  PD3   = SD-Card D_in (MOSI) (voltage divider 1k/2.2k) [Mega16: + JP1]
;  PD4   = SD-Card D_out (MISO) (resistor 1k)
;  PD5   = SD-Card Clock (voltage divider 1k/2.2k) [Mega16: + JP2]
;  PD6/7 = LED Col 18/17 (MSB left)
;  PE0   = SD-Card Detect Switch (low = card is present) [not on Mega16]
;  PE1/2 = Baud Rate jumpers JP1/JP2 to GND [not on Mega16 or BSA-Plus PCB Rev1]
;
;  Mega16: Jumpers JP1 and JP2 each have a 1k or 1.2k resistor in series with
;          the jumper and are connected between PD3/PD5 and +5V. See BlinkenLEDs
;          Plus Rev. 1 schematic for an example.
;
;Frame Timing:
;  Controller frequency : 14.7456 MHz
;  Timer prescaler      : 256
;  PWM "steps" per row  : 72
;  Multiplexing         : 8 rows
;    => 14.7456 MHz / 256 / 72 / 8 = 100 Hz (error: 0 %)
;
;Example PWM durations (for 144-pixel projects except BlinkstroemAdvanced):
;0, 1, 3, 6, 11, 21, 39, 72  - exponential, base 1.8
;0, 2, 5, 8, 12, 21, 39, 72  - equivalent to old optimized safe exponential
;0, 1, 6, 13, 24, 37, 53, 72 - Gamma 2.0 (y = x^2 * 72/49, x = [0..7])
;0, 2, 6, 13, 24, 37, 53, 72 - Safe Gamma 2.0 (delta >= 2)
;0, 1, 3, 9, 18, 31, 49, 72  - Gamma 2.5
;absolute minimum step between two values:    1
;recommended minimum step between two values: 2 ("safe")

.equ	PWMVAL_0 = 0 ;don't change this value
.equ	PWMVAL_1 = 2
.equ	PWMVAL_2 = 5
.equ	PWMVAL_3 = 8
.equ	PWMVAL_4 = 12
.equ	PWMVAL_5 = 21
.equ	PWMVAL_6 = 39
.equ	PWMVAL_7 = 72 ;don't change this value (timing depends on it)

;===============================================================================

;SD card pins
.equ	USE_HARDWARE_SPI = 0
.equ	SD_PORT = PORTD
.equ	SD_PIN = PIND
.equ	SD_CS = 2
.equ	SD_MOSI = 3
.equ	SD_MISO = 4
.equ	SD_CK = 5
.if (MCU_HAS_PORTE)
  .equ	SDCARD_HAS_DETECT_PIN = 1
  .equ	SD_DETECT_PIN = PINE
  .equ	SD_DETECT = 0 ;Card Detect switch pin on PORTE
.else
  .equ	SDCARD_HAS_DETECT_PIN = 0
.endif

;other settings
.equ	OUT_TIMING_DIV = 256
.equ	DISALLOW_UART = 0
.equ	DISALLOW_UART_TX = 0

;===============================================================================

.macro _set_led_1 ;LEDs 1 to 8
	ld	temp, Z+
	swap	temp
	cp	temp, pwm ;carry set if pwm > temp
  .if (OUT_REVERSE_PORTA)
	rol	R0
  .else
	ror	R0
  .endif
	swap	temp
	cp	temp, pwm ;carry set if pwm > temp
  .if (OUT_REVERSE_PORTA)
	rol	R0
  .else
	ror	R0
  .endif
.endmacro
	
.macro _set_led_2 ;LEDs 9 to 16
	ld	temp, Z+
	swap	temp
	cp	temp, pwm ;carry set if pwm > temp
	rol	R1
	swap	temp
	cp	temp, pwm ;carry set if pwm > temp
	rol	R1
.endmacro

.macro _set_led_3 ;LEDs 17 and 18
	ld	temp, Z+
	cp	temp, pwm
	brlo	PC+2
  .if (OUT_INVERT_COLUMNS)
	andi	temp2, LOW(~(@1)) ;'LOW' prevents assembler warning
  .else
	ori	temp2, @1
  .endif
	swap	temp
	cp	temp, pwm
	brlo	PC+2
  .if (OUT_INVERT_COLUMNS)
	andi	temp2, LOW(~(@0)) ;'LOW' prevents assembler warning
  .else
	ori	temp2, @0
  .endif
.endmacro

;===============================================================================
	
oc0:	;Timer 0 output compare interrupt (PWM steps)
	push	temp
	push	temp2
	_push_w	Z
	push	R0
	push	R1
	in	sreg_backup, SREG
	
	;increment row
	lsl	mux
	brne	oc0_rowinc_end
	ldi	mux, 0x01
	;increment PWM value ('pwm' register stores pwm value in high nibble!)
	subi	pwm, -0x30 ;PWM value sequence: 3, 6, 1, 4, 7, 2, 5, (0)
	andi	pwm, 0x70
	brne	oc0_pwminc_end ;zero: end of PWM sequence
	ldi	pwm, 0x30 ;start with first value of PWM sequence
	tick_100hz
oc0_pwminc_end:
	;load start address
	activeframe Z ;max. 6 cycles
	_sts_w	RAM_MuxAddress, Z
oc0_rowinc_end:
	
	;determine next interval
	mov	temp, pwm
	swap	temp
	_ldi_w	Z, (FLASH_OFFSET + pwm_duration) * 2
	add	ZL, temp
	adc	ZH, zero
	lpm	temp, Z
	out	OCR0, temp
	
	;load framebuffer address for current row
	_lds_w	Z, RAM_MuxAddress
	
	;set LEDs 1 to 8
	_set_led_1 ;8 cycles each
	_set_led_1
	_set_led_1
	_set_led_1
	
	;set LEDs 9 to 16
	_set_led_2 ;8 cycles each
	_set_led_2
	_set_led_2
	_set_led_2
	
	;set LEDs 17, 18
	in	temp2, PORTD
.if (OUT_INVERT_COLUMNS)
	ori	temp2, 0xC0
.else
	andi	temp2, 0x3F
	com	R0
	com	R1
.endif
	_set_led_3 0x80, 0x40 ;9 cycles
	
.if (OUT_BLPLUS_REV1_CORRECTION)
	;correction for PCB layout error in BlinkenLEDs Plus Kit Rev. 1
	;map bits [76543210] => [23456710]
	mov	temp, R0
	swap	R0
	lsl	R0
	lsl	R0
	lsl	temp
	ror	R0
	lsl	temp
	ror	R0
	lsl	temp
	ror	R0
	lsl	temp
	ror	R0
	lsl	temp
	ror	R0
	lsl	temp
	ror	R0
.endif
	
	;old row off
	out	PORTB, zero
	
	;delay until FET is off
	_sts_w	RAM_MuxAddress, Z ;store framebuffer address for next row
	
	;output new LED data
	out	PORTA, R0
	out	PORTC, R1
	out	PORTD, temp2
	
	;delay until transistors have switched
	out	SREG, sreg_backup
	pop	R1
	pop	R0
	_pop_w	Z
	pop	temp2
	pop	temp
	
	;new row on
.if (OUT_INVERT_ROWS)
	com	mux
.endif
	out	PORTB, mux
.if (OUT_INVERT_ROWS)
	com	mux
.endif

	;return
	reti

;===============================================================================

.macro init_output
	
	;initialize registers
	ldi	mux, 0x00 ;no row
	ldi	pwm, 0x50 ;last PWM value in sequence

	;initialize ports
	ldi	temp, 0xFF
	out	DDRA, temp
	out	DDRC, temp
  .if (OUT_INVERT_COLUMNS)
	ldi	temp, 0xFF
  .else
	ldi	temp, 0x00
  .endif
	out	PORTA, temp
	out	PORTC, temp

	ldi	temp, 0xFF
	out	DDRB, temp
  .if (OUT_INVERT_ROWS)
	ldi	temp, 0xFF
  .else
	ldi	temp, 0x00
  .endif
	out	PORTB, temp
	
  .if (JUMPERS && (MCU_HAS_PORTE == 0) && (STREAM_INPUT||STREAM_OUTPUT||DEBUG))

	;read jumpers on Mega16: set MOSI and CK to input
	;                        (use external voltage dividers as pull-downs)
	ldi	temp, 0xC3 | 1<<SD_CS
	out	DDRD, temp
    .if (OUT_INVERT_COLUMNS)
	ldi	temp, 0xC1 | 1<<SD_CS | 1<<SD_MISO ;read jumpers: no pullups
    .else
	ldi	temp, 0x01 | 1<<SD_CS | 1<<SD_MISO ;read jumpers: no pullups
    .endif
	out	PORTD, temp
	
  .else

	;no jumpers or jumpers on PORTE
	ldi	temp, 0xC3 | 1<<SD_CS | 1<<SD_MOSI | 1<<SD_CK
	out	DDRD, temp
    .if (OUT_INVERT_COLUMNS)
	ldi	temp, 0xC1 | 1<<SD_CS | 1<<SD_MOSI | 1<<SD_MISO
    .else
	ldi	temp, 0x01 | 1<<SD_CS | 1<<SD_MOSI | 1<<SD_MISO
    .endif
	out	PORTD, temp

  .endif

  .if (MCU_HAS_PORTE)
	ldi	temp, 0x00
	out	DDRE, temp
	ldi	temp, 0x07
	out	PORTE, temp
  .endif

.endmacro



.macro init_output_jumpers

  .if (JUMPERS && (STREAM_INPUT || STREAM_OUTPUT || DEBUG))

    .if (MCU_HAS_PORTE == 0)
	
	;read baud rate jumpers on PIND (shared pins with SD card)
	in	temp2, PIND
	andi	temp2, 0x28
      .equ JUMPERS_JP1_CLOSED = 0x08
      .equ JUMPERS_JP2_CLOSED = 0x20
      .equ JUMPERS_BOTH_CLOSED = 0x28
      .if (OUT_INVERT_COLUMNS)
	ldi	temp, 0xC1 | 1<<SD_CS | 1<<SD_MOSI | 1<<SD_MISO
      .else
	ldi	temp, 0x01 | 1<<SD_CS | 1<<SD_MOSI | 1<<SD_MISO
      .endif
	out	PORTD, temp
	ldi	temp, 0xC3 | 1<<SD_CS | 1<<SD_MOSI | 1<<SD_CK
	out	DDRD, temp
	
    .else
	
	;read baud rate jumpers on PINE
	in	temp2, PINE
	andi	temp2, 0x06
      .equ JUMPERS_JP1_CLOSED = 0x04
      .equ JUMPERS_JP2_CLOSED = 0x02
      .equ JUMPERS_BOTH_CLOSED = 0x00

  .endif

.endmacro

