#!/usr/bin/perl -w
use Time::HiRes "usleep";

if (scalar(@ARGV) < 3) {
	die "Usage: $0 <PORT>[:<BAUD>] <WIDTH> <HEIGHT> [<CHANNELS> [<OPTION>]]\n" .
	"\n" .
	"  <PORT>      Serial port device (e.g. /dev/ttyS0 or /dev/ttyUSB0)\n" .
	"  <BAUD>      Baudrate (optional, default 115200)\n" .
	"  <WIDTH>     Display width in pixels (e.g. 18)\n" .
	"  <HEIGHT>    Display height in pixels (e.g. 8)\n" .
	"  <CHANNELS>  Display color depth (optional, default 1):\n" .
	"              1 = grayscale, 2 = RG, 3 = RGB\n" .
	"  <OPTION>    Additional options (comma-separated):\n" .
	"                * single     Only emit a single frame and exit\n" .
	"\n";
	
}

$PORT = $ARGV[0];
if ($PORT =~ /^[^:]+:[0-9]+$/) {
	$PORT = $1;
	$BAUD = $2;
} else {
	$BAUD = 115200;
}
$WIDTH = $ARGV[1];
$HEIGHT = $ARGV[2];
if (scalar(@ARGV) >= 4) {
	$CHANNELS = $ARGV[3];
} else {
	$CHANNELS = 1;
}

$SINGLE = 0;
if (scalar(@ARGV) >= 5) {
	for (split(',', $ARGV[4])) {
		if (/single/i) {
			$SINGLE = 1;
		}
	}
}


system "stty -F $PORT $BAUD raw -echo";
open SER, ">$PORT";
$x = select SER;
$| = 1;
select $x;

# time per frame: time required to send data + 5 ms (safety margin)
$time = 1e6 * ($WIDTH * $HEIGHT * $CHANNELS + 12) / ($BAUD / 10) + 5000;
$time = 50000 if ($time < 50000);

print "Sending stream: ${WIDTH}x${HEIGHT}-${CHANNELS}/8 (",
	int($time / 1000), " ms/frame)\n";

if ($CHANNELS == 3) {
	$start = 0x700; # red
	$dir = 0x010;
} else {
	$start = 0x000; # black
	$dir = 0x100;
}

while (1) {
	print SER "\x23\x54\x26\x66", pack('nnn', $HEIGHT, $WIDTH, $CHANNELS), "\x00\x07";
	$row = $start;
	$row_dir = $dir;
	for ($y = 0; $y < $HEIGHT; $y++) {
		$pixel = $row;
		$pixel_dir = $row_dir;
		for ($x = 0; $x < $WIDTH; $x++) {
			print SER pack("C" x $CHANNELS, ($pixel >> 8), ($pixel >> 4) & 0x07,
			               $pixel & 0x07);
			$pixel += $pixel_dir;
			$pixel_dir = new_add($pixel, $pixel_dir);
		}
		$row += $row_dir;
		$row_dir = new_add($row, $row_dir);
	}
	$start += $dir;
	$dir = new_add($start, $dir);
	exit if ($SINGLE);
	usleep ($time);
}

sub new_add {
	my ($val, $add) = @_;
	if ($val == 0x000) {
		$add = 0x100;
	} elsif ($val == 0x700) {
		$add = ($CHANNELS == 1) ? -0x100 : 0x010;
	} elsif ($val == 0x770) {
		$add = -0x100;
	} elsif ($val == 0x070) {
		$add = ($CHANNELS == 2) ? -0x010 : 0x001;
	} elsif ($val == 0x077) {
		$add = -0x010;
	} elsif ($val == 0x007) {
		$add = 0x100;
	} elsif ($val == 0x707) {
		$add = -0x001;
	}
	return $add;
}
